# Aerial Vision - Professional 3D Model Showcase Platform

A production-ready web platform for showcasing drone-captured 3D models, similar to Nira.app. Features cloud storage, shareable links, and professional 3D rendering.

![Aerial Vision Preview](preview.png)

## ✨ Features

### Core Functionality
- ✅ **Cloud File Storage** - Upload models to server, not just browser
- ✅ **Shareable Links** - Each model gets a unique shareable URL
- ✅ **Multi-Format Support** - GLTF, GLB, OBJ, FBX, PLY, LAS/LAZ, SPLAT
- ✅ **Database Storage** - SQLite database for models and metadata
- ✅ **RESTful API** - Full backend API for model management
- ✅ **Public/Private Viewing** - Share models via unique links

### 3D Rendering
- 🎨 **Textured Meshes** - Full GLTF/GLB/OBJ/FBX support
- ☁️ **Point Clouds** - PLY, LAS, LAZ with proper rendering
- 🌟 **Gaussian Splats** - SPLAT format support
- 🔄 **Progressive Loading** - Handles large models efficiently
- 🎛️ **Interactive Controls** - Orbit, pan, zoom, measure

### User Experience
- 📱 **Responsive Design** - Works on desktop, tablet, mobile
- 🎨 **Modern UI** - Clean, professional interface
- ⚡ **Fast Performance** - Optimized for large models
- 🔗 **Share Links** - Copy shareable URLs instantly

## 🚀 Quick Start

### Prerequisites
- Node.js 18+ installed
- npm or yarn

### Installation

1. **Install frontend dependencies**
   ```bash
   npm install
   ```

2. **Install backend dependencies**
   ```bash
   cd server
   npm install
   cd ..
   ```

3. **Start the backend server**
   ```bash
   cd server
   npm run dev
   ```
   Server runs on `http://localhost:3001`

4. **Start the frontend** (in a new terminal)
   ```bash
   npm run dev
   ```
   Frontend runs on `http://localhost:5173`

### Production Build

1. **Build frontend**
   ```bash
   npm run build
   ```

2. **Start production server**
   ```bash
   cd server
   npm start
   ```

The server will serve the built frontend from the `dist` folder.

## 📁 Project Structure

```
aerial-vision/
├── server/                 # Backend API
│   ├── server.js          # Express server
│   ├── uploads/           # Stored model files
│   └── database.sqlite    # SQLite database
├── src/                    # Frontend React app
│   ├── components/         # React components
│   ├── api/               # API client
│   └── store/             # State management
├── dist/                   # Built frontend (after npm run build)
└── package.json
```

## 🌐 Deployment

### Option 1: Single Server Deployment

1. Build frontend: `npm run build`
2. Copy `dist/` folder to server
3. Copy `server/` folder to server
4. Install server dependencies: `cd server && npm install`
5. Start server: `npm start`
6. Configure reverse proxy (Nginx/Apache) to point to port 3001

### Option 2: Separate Frontend/Backend

**Backend:**
- Deploy `server/` folder to your server
- Set `PORT` environment variable
- Ensure `uploads/` directory is writable

**Frontend:**
- Build: `npm run build`
- Deploy `dist/` to static hosting (Vercel, Netlify, etc.)
- Set `VITE_API_URL` environment variable to your backend URL

### Nginx Configuration

```nginx
server {
    listen 80;
    server_name yourdomain.com;

    # Serve frontend
    location / {
        root /path/to/dist;
        try_files $uri $uri/ /index.html;
    }

    # Proxy API requests
    location /api {
        proxy_pass http://localhost:3001;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }

    # Serve model files
    location /api/models/ {
        proxy_pass http://localhost:3001;
        proxy_set_header Host $host;
        add_header Access-Control-Allow-Origin *;
    }
}
```

## 🔧 Configuration

### Environment Variables

Create `.env` file in root:

```env
VITE_API_URL=http://localhost:3001/api
PORT=3001
```

For production, set `VITE_API_URL` to your backend URL.

## 📡 API Endpoints

### Models

- `GET /api/models` - Get all models
- `GET /api/models/:id` - Get single model
- `POST /api/models` - Upload model (multipart/form-data)
- `DELETE /api/models/:id` - Delete model
- `GET /api/models/:id/file` - Download model file

### Share Links

- `GET /view/:shareId` - View shared model (public)

## 🎮 Usage

1. **Upload Models**: Drag & drop files or click to browse
2. **View Models**: Click any model card to view in 3D
3. **Share Models**: Copy the shareable link from model details
4. **Manage Models**: Delete models from the gallery

## 🛠️ Technologies

**Backend:**
- Express.js - Web server
- SQLite (better-sqlite3) - Database
- Multer - File uploads
- UUID - Unique IDs

**Frontend:**
- React 18 - UI framework
- Three.js - 3D rendering
- @react-three/fiber - React renderer
- Zustand - State management
- Tailwind CSS - Styling
- React Router - Routing

## 📝 License

MIT License - feel free to use for personal or commercial projects.

---

Built for drone enthusiasts and 3D scanning professionals. 🚁✨
